DROP PROCEDURE [hips].[HospitalPatientGet]
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
-- =============================================
-- Author:        Gordon Massey
-- Create date:   15 October 2012
-- Description:   Gets a specified record from the HospitalPatient table.
-- Modified:
-- 04/06/2013 - Simon Biber - Add ability to get only active records.
--              When the ActiveOnly parameter is true, the MRN that has an
--              episode with the most recent (most future) admission date/time
--              will be chosen from among those with the given hospital and
--              patient master ID.
-- =============================================
CREATE PROCEDURE [hips].[HospitalPatientGet] 
(
	@PatientId INT = null,
	@HospitalId INT = NULL,
	@Mrn VARCHAR(20) = NULL,
	@PatientMasterId INT = NULL,
	@ActiveOnly BIT = NULL	
)
AS
BEGIN
	IF (@ActiveOnly = '1')
	BEGIN
		-- Get only the MRN associated with the latest episode for each patient at each hospital.
		-- In 99% of cases this will return only one MRN for each patient at each hospital.
		-- The only case where it will return more than one is if there are episodes with the
		-- exact same admission date/time on multiple MRNs. In this case they are ordered with the
		-- most recently modified HospitalPatient record first.
		SELECT 
			hp.PatientId,
			hp.PatientMasterId,
			hp.HospitalId,
			hp.Mrn,
			hp.DateCreated,
			hp.UserCreated,
			hp.DateModified,
			hp.UserModified
		FROM hips.HospitalPatient hp
		INNER JOIN hips.Episode e
			ON hp.PatientId = e.PatientId
		INNER JOIN
				(	-- The latest admission date/time for the patient master at the hospital
					SELECT  PatientMasterId, HospitalId, MAX(AdmissionDate) AdmissionDate
					FROM    hips.HospitalPatient hp
					INNER JOIN hips.Episode e
						ON hp.PatientId = e.PatientId
					WHERE  (@PatientMasterId IS NULL OR PatientMasterId = @PatientMasterId)
					AND    (@HospitalId IS NULL OR HospitalId = @HospitalId)
					GROUP BY PatientMasterId, HospitalId
				) AS LatestAdmission
			ON hp.PatientMasterId = LatestAdmission.PatientMasterId
			AND	hp.HospitalId = LatestAdmission.HospitalId
			AND e.AdmissionDate = LatestAdmission.AdmissionDate
		WHERE   (@PatientId = hp.PatientId OR @PatientId is NULL)
			AND (@HospitalId = hp.HospitalId OR @HospitalId is NULL)
			AND (@Mrn = hp.Mrn OR @Mrn is NULL)
			AND (@PatientMasterId = hp.PatientMasterId OR @PatientMasterId  IS NULL)
			AND (@HospitalId = hp.HospitalId OR @HospitalId IS NULL)
		ORDER BY hp.DateModified DESC
	END
	ELSE
	BEGIN
		-- Get all the MRNs, including active and inactive MRNs, for the specified
		-- patient and hospital.
		SELECT
			PatientId,
			PatientMasterId,
			HospitalId,
			Mrn,
			DateCreated,
			UserCreated,
			DateModified,
			UserModified
		FROM hips.HospitalPatient
		WHERE   (@PatientId = PatientId OR @PatientId is NULL)
			AND (@HospitalId = HospitalId OR @HospitalId is NULL)
			AND (@Mrn = Mrn OR @Mrn is NULL)
			AND (@PatientMasterId = PatientMasterId OR @PatientMasterId  IS NULL)
		ORDER BY DateModified DESC
	END
END
GO

